<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class CollegeFeedbackSurvey extends CI_Controller {

    public function __construct()
    {
        parent::__construct();
        $this->load->model('CollegeFeedbackSurvey_model');
        $this->load->helper('url');
    }

    /**
     * Main survey create page
     * URL format: /admin/CollegeFeedbackSurvey/create/instituteId/feedbackforid/branchid/year
     */
    public function create($institute_id = null, $feedbackfor_id = null, $branch_id = null, $financial_year_id = null)
    {
        // ==================== VALIDATE URL PARAMETERS ====================
        
        if (empty($institute_id) || empty($feedbackfor_id) || empty($financial_year_id)) {
            $this->show_error_page(
                'Invalid Parameters', 
                'Required parameters are missing. Please check the URL and try again.'
            );
            return;
        }

        if ($branch_id === null) {
            $branch_id = 0;
        }

        // ==================== VALIDATE URL FROM collegefeedback_master ====================
        
        $url_data = $this->CollegeFeedbackSurvey_model->validate_survey_url(
            $institute_id, 
            $feedbackfor_id, 
            $branch_id, 
            $financial_year_id
        );

        if (!$url_data || $url_data->is_active == 0) {
            $this->show_error_page(
                'URL Not Found', 
                'The feedback URL you are trying to access is either invalid or has been deactivated. Please contact your administrator.'
            );
            return;
        }

        $fkURL_id = $url_data->url_id;

        // ==================== FETCH MASTER DATA ====================

        $heading_data = $this->CollegeFeedbackSurvey_model->get_heading_data();
        $financial_year = $this->CollegeFeedbackSurvey_model->get_financial_year($financial_year_id);
        
        if (!$financial_year) {
            $this->show_error_page(
                'Invalid Financial Year', 
                'The financial year specified in the URL is invalid. Please check the URL and try again.'
            );
            return;
        }

        $institute_data = $this->CollegeFeedbackSurvey_model->get_institute($institute_id);
        if (!$institute_data) {
            $this->show_error_page(
                'Invalid Institute', 
                'The institute specified in the URL is invalid. Please check the URL and try again.'
            );
            return;
        }

        $feedbackfor_data = $this->CollegeFeedbackSurvey_model->get_feedbackfor_name($feedbackfor_id);
        if (!$feedbackfor_data) {
            $this->show_error_page(
                'Invalid Feedback Type', 
                'The feedback type specified in the URL is invalid. Please check the URL and try again.'
            );
            return;
        }

        $branch_data = null;
        $is_branch_fixed = false;

        if ($branch_id != 0) {
            $branch_data = $this->CollegeFeedbackSurvey_model->get_branch($branch_id);
            if (!$branch_data) {
                $this->show_error_page(
                    'Invalid Branch', 
                    'The branch specified in the URL is invalid. Please check the URL and try again.'
                );
                return;
            }
            $is_branch_fixed = true;
        }

        $branches = $this->CollegeFeedbackSurvey_model->get_all_branches($institute_id);

        // ==================== FETCH QUESTIONS & ANSWERS ====================

        $questions = $this->CollegeFeedbackSurvey_model->get_questions_by_feedback($feedbackfor_id);
        $answer_options = $this->CollegeFeedbackSurvey_model->get_answer_options($feedbackfor_id);

      // After getting the feedbackfor_data, add this line:
 $questionperpage = isset($feedbackfor_data->questionperpage) ? $feedbackfor_data->questionperpage : 10; // Default to 10 if not set

// Update the data array to include questionperpage
 $data = [
    'heading'            => $heading_data,
    'financial_year'     => $financial_year,
    'institute_id'       => $institute_data->Institu_id,
    'institute_name'     => $institute_data->Institu_Name,
    'feedbackfor_id'     => $feedbackfor_data->feedbackid,
    'feedbackfor_name'   => $feedbackfor_data->feedbackforname,
    'branch_id'          => $branch_id,
    'branch_name'        => $branch_data ? $branch_data->branch_name : null,
    'is_branch_fixed'    => $is_branch_fixed,
    'branches'           => $branches,
    'fkURL_id'           => $fkURL_id,
    'questions'          => $questions,
    'answer_options'     => $answer_options,
    'questionperpage'    => $questionperpage // Add this line
];

        $this->load->view('admin/CollegeFeedbackSurvey_view', $data);
    }

    public function submit_survey()
    {
        // ==================== GET POSTED DATA ====================
        $branch = $this->input->post('branch');
        $financial_year_id = $this->input->post('financial_year_id');
        $institute_id = $this->input->post('institute_id');
        $feedbackfor_id = $this->input->post('feedbackfor_id');
        $fkURL_id = $this->input->post('fkURL_id');
        $branch_id_from_url = $this->input->post('branch_id_from_url');

        // ==================== VERIFY URL STATUS ====================
        if (!empty($fkURL_id)) {
            $url_check = $this->db->select('is_active')
                                  ->from('collegefeedback_master')
                                  ->where('url_id', $fkURL_id)
                                  ->get()->row();

            if (!$url_check || $url_check->is_active == 0) {
                echo json_encode(['status' => 'error', 'message' => 'Feedback URL is inactive.']);
                return;
            }
        }

        // ==================== GET BRANCH ID ====================
        $branch_id = (!empty($branch_id_from_url) && $branch_id_from_url != 0) ? $branch_id_from_url : 0;
        if ($branch_id == 0 && !empty($branch)) {
            $branch_data = $this->db->select('branch_id')
                                    ->from('branch_master')
                                    ->where('branch_name', $branch)
                                    ->get()->row();
            $branch_id = $branch_data ? $branch_data->branch_id : 0;
        }

        // ==================== GET QUESTIONS TO DETERMINE ANSWER TYPE ====================
        $questions = $this->CollegeFeedbackSurvey_model->get_questions_by_feedback($feedbackfor_id);
        $question_types = [];
        foreach ($questions as $question) {
            $question_types[$question->questionid] = $question->isanswerortext;
        }

        // ==================== PREPARE RESPONSES ====================
        $responses = [];
        $all_post = $this->input->post();
        $current_date = date('Y-m-d H:i:s');

        foreach ($all_post as $key => $value) {
            // Checking for keys like q1, q2...
            if (strpos($key, 'q') === 0 && is_numeric(substr($key, 1))) {
                $question_id = str_replace('q', '', $key);
                
                if (!isset($question_types[$question_id])) {
                    continue;
                }
                
                $answer_type = $question_types[$question_id];
                
                // Base response data
                $response_base = [
                    'questionid'    => $question_id,
                    'feedbackforid' => $feedbackfor_id,
                    'fkinstituteid' => $institute_id,
                    'fkbranch_id'   => $branch_id,
                    'finicial_Year' => $financial_year_id,
                    'is_active'     => 1,
                    'url_id'        => $fkURL_id
                ];
                
                // Handle different answer types
                switch ($answer_type) {
                    case 1: // Radio button
                        $response_base['answertypeid'] = $value;
                        $response_base['answer'] = '';
                        $responses[] = $response_base;
                        break;
                        
                    case 2: // Checkbox - single selection (behaves like radio)
                        $response_base['answertypeid'] = $value;
                        $response_base['answer'] = '';
                        $responses[] = $response_base;
                        break;
                        
                    case 3: // Select2 dropdown
                        $response_base['answertypeid'] = $value;
                        $response_base['answer'] = '';
                        $responses[] = $response_base;
                        break;
                        
                    case 4: // Textarea
                        $response_base['answertypeid'] = 0;
                        $response_base['answer'] = $value;
                        $responses[] = $response_base;
                        break;
                }
            }
        }

        // ==================== INSERT & RESPONSE ====================
        if (empty($responses)) {
            echo json_encode(['status' => 'error', 'message' => 'No responses to save.']);
            return;
        }

        $insert_result = $this->CollegeFeedbackSurvey_model->insert_survey_responses($responses);

        if ($insert_result) {
            echo json_encode(['status' => 'success', 'message' => 'Survey submitted successfully!']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Database insertion failed.']);
        }
    }

    private function show_error_page($title, $message)
    {
        $error_html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . $title . '</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: "Segoe UI", Tahoma, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .error-container {
            background: white;
            padding: 40px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
            text-align: center;
            max-width: 500px;
            width: 100%;
        }
        .error-icon {
            width: 80px;
            height: 80px;
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            border-radius: 50%;
            margin: 0 auto 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 40px;
            color: white;
        }
        .error-container h1 {
            font-size: 24px;
            color: #333;
            margin-bottom: 15px;
            font-weight: 600;
        }
        .error-container p {
            color: #666;
            font-size: 16px;
            line-height: 1.6;
            margin-bottom: 25px;
        }
        .contact-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
            font-size: 14px;
            color: #666;
            margin-top: 20px;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-icon">⚠️</div>
        <h1>' . $title . '</h1>
        <p>' . $message . '</p>
        <div class="contact-info">
            <strong>Need Help?</strong><br>
            Please contact your college administrator for assistance.
        </div>
    </div>
</body>
</html>';
        
        echo $error_html;
        exit;
    }
}
?>