<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class File_Download extends CI_Controller {
    
    public function __construct()
    {
        parent::__construct();
        $this->load->model('File_Download_model');
        $this->load->helper(['url', 'download']);
        $this->load->library('zip');
        
        // Set CORS headers
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
    }

    public function index()
    {
        $this->load->view('admin/header_view');
        $this->load->view('admin/File_Download_view');
        $this->load->view('admin/footer_view');
    }

    public function get_files_list()
    {
        header('Content-Type: application/json');
        
        try {
            $start_date = $this->input->post('start_date');
            $end_date = $this->input->post('end_date');
            
            if (empty($start_date) || empty($end_date)) {
                echo json_encode([
                    'status' => 'error', 
                    'message' => 'Please select both start and end dates'
                ]);
                return;
            }

            // Get files from uploads folder based on date range
            $files = $this->File_Download_model->getFilesByDateRange($start_date, $end_date);
            
            if (empty($files)) {
                echo json_encode([
                    'status' => 'error', 
                    'message' => 'No files found for the selected date range'
                ]);
                return;
            }

            // Prepare file list with download URLs
            $file_list = [];
            foreach ($files as $file) {
                if (file_exists($file['path'])) {
                    // URL-safe base64 encoding
                    $encoded_path = $this->url_safe_encode($file['relative_path']);
                    
                    $file_list[] = [
                        'name' => $file['name'],
                        'size' => $this->formatBytes($file['size']),
                        'modified' => $file['modified'],
                        'path' => $file['relative_path'],
                        'folder_name' => $file['folder_name'],
                        'download_url' => base_url('admin/File_Download/download_single_file/' . $encoded_path)
                    ];
                }
            }

            echo json_encode([
                'status' => 'success',
                'file_count' => count($file_list),
                'files' => $file_list
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Get files list error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error', 
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    public function download_single_file($encoded_path = '')
    {
        try {
            if (empty($encoded_path)) {
                log_message('error', 'Empty encoded path');
                show_error('File path is missing', 404);
                return;
            }

            // URL-safe base64 decode
            $relative_path = $this->url_safe_decode($encoded_path);
            
            if (empty($relative_path)) {
                log_message('error', 'Failed to decode path: ' . $encoded_path);
                show_error('Invalid file path', 404);
                return;
            }

            // Normalize path separators to forward slashes
            $relative_path = str_replace('\\', '/', $relative_path);
            $relative_path = ltrim($relative_path, '/');
            
            // Build the full file path
            $uploads_base = FCPATH . 'uploads/';
            $file_path = $uploads_base . $relative_path;
            
            log_message('info', 'Attempting download - Relative: ' . $relative_path);
            log_message('info', 'Full path: ' . $file_path);
            
            // Check if file exists
            if (!file_exists($file_path)) {
                log_message('error', 'File not found: ' . $file_path);
                show_error('File not found: ' . basename($relative_path), 404);
                return;
            }
            
            if (!is_file($file_path)) {
                log_message('error', 'Path is not a file: ' . $file_path);
                show_error('Invalid file path', 404);
                return;
            }
            
            // Check if file is readable
            if (!is_readable($file_path)) {
                log_message('error', 'File is not readable: ' . $file_path);
                show_error('File is not accessible', 403);
                return;
            }
            
            // Security check - ensure file is within uploads directory
            $real_file_path = realpath($file_path);
            $real_uploads_path = realpath($uploads_base);
            
            if ($real_file_path === false) {
                log_message('error', 'Could not resolve real path for: ' . $file_path);
                // If realpath fails, do a manual check
                $normalized_file = str_replace('\\', '/', $file_path);
                $normalized_uploads = str_replace('\\', '/', $uploads_base);
                
                if (strpos($normalized_file, $normalized_uploads) !== 0) {
                    log_message('error', 'Security violation - file outside uploads');
                    show_error('Access denied', 403);
                    return;
                }
                
                // Use the original path if realpath failed but manual check passed
                $real_file_path = $file_path;
            } else {
                // Normalize the real paths
                $real_file_path = str_replace('\\', '/', $real_file_path);
                $real_uploads_path = str_replace('\\', '/', $real_uploads_path);
                
                // Security check
                if (strpos($real_file_path, $real_uploads_path) !== 0) {
                    log_message('error', 'Security violation: ' . $real_file_path . ' is outside ' . $real_uploads_path);
                    show_error('Access denied', 403);
                    return;
                }
            }
            
            $file_name = basename($real_file_path);
            $file_size = filesize($real_file_path);
            
            log_message('info', 'Starting download: ' . $file_name . ' (' . $file_size . ' bytes)');
            
            // Read file content
            $file_content = file_get_contents($real_file_path);
            
            if ($file_content === false) {
                log_message('error', 'Failed to read file: ' . $real_file_path);
                show_error('Failed to read file', 500);
                return;
            }
            
            // Use CodeIgniter's force_download helper
            force_download($file_name, $file_content);
            
            log_message('info', 'Download completed: ' . $file_name);
            
        } catch (Exception $e) {
            log_message('error', 'Download exception: ' . $e->getMessage() . ' | Line: ' . $e->getLine());
            show_error('Download failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * URL-safe base64 encoding
     */
    private function url_safe_encode($string)
    {
        return strtr(base64_encode($string), '+/=', '-_~');
    }

    /**
     * URL-safe base64 decoding
     */
    private function url_safe_decode($string)
    {
        return base64_decode(strtr($string, '-_~', '+/='));
    }

    /**
     * Download file using POST method (better for special characters)
     */
    public function download_single_file_post()
    {
        try {
            $relative_path = $this->input->post('file_path');
            
            if (empty($relative_path)) {
                log_message('error', 'Empty file path in POST');
                $this->output->set_status_header(404);
                echo json_encode(['error' => 'File path is missing']);
                return;
            }

            // Normalize path separators
            $relative_path = str_replace('\\', '/', $relative_path);
            $relative_path = ltrim($relative_path, '/');
            
            // Build full file path
            $uploads_base = FCPATH . 'uploads/';
            $file_path = $uploads_base . $relative_path;
            
            log_message('info', 'POST Download - Relative: ' . $relative_path);
            log_message('info', 'POST Download - Full path: ' . $file_path);
            
            // Validate file exists
            if (!file_exists($file_path) || !is_file($file_path)) {
                log_message('error', 'File not found: ' . $file_path);
                $this->output->set_status_header(404);
                echo json_encode(['error' => 'File not found']);
                return;
            }
            
            // Check readable
            if (!is_readable($file_path)) {
                log_message('error', 'File not readable: ' . $file_path);
                $this->output->set_status_header(403);
                echo json_encode(['error' => 'File not accessible']);
                return;
            }
            
            // Security check
            $real_file_path = realpath($file_path);
            $real_uploads_path = realpath($uploads_base);
            
            if ($real_file_path === false) {
                $normalized_file = str_replace('\\', '/', $file_path);
                $normalized_uploads = str_replace('\\', '/', $uploads_base);
                
                if (strpos($normalized_file, $normalized_uploads) !== 0) {
                    log_message('error', 'Security violation in POST download');
                    $this->output->set_status_header(403);
                    echo json_encode(['error' => 'Access denied']);
                    return;
                }
                $real_file_path = $file_path;
            } else {
                $real_file_path = str_replace('\\', '/', $real_file_path);
                $real_uploads_path = str_replace('\\', '/', $real_uploads_path);
                
                if (strpos($real_file_path, $real_uploads_path) !== 0) {
                    log_message('error', 'Security violation: ' . $real_file_path);
                    $this->output->set_status_header(403);
                    echo json_encode(['error' => 'Access denied']);
                    return;
                }
            }
            
            $file_name = basename($real_file_path);
            $file_size = filesize($real_file_path);
            
            log_message('info', 'POST Download starting: ' . $file_name);
            
            // Read file
            $file_content = file_get_contents($real_file_path);
            
            if ($file_content === false) {
                log_message('error', 'Failed to read file: ' . $real_file_path);
                $this->output->set_status_header(500);
                echo json_encode(['error' => 'Failed to read file']);
                return;
            }
            
            // Force download
            force_download($file_name, $file_content);
            
            log_message('info', 'POST Download completed: ' . $file_name);
            
        } catch (Exception $e) {
            log_message('error', 'POST Download exception: ' . $e->getMessage());
            $this->output->set_status_header(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    private function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}