<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class StudentFeedback_list_model extends CI_Model {
    
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }
    
    // ==================== STUDENT FEEDBACK MASTER TABLE FUNCTIONS ====================
    
    /**
     * Insert student feedback record
     */
    public function insertStudent($data)
    {
        $this->db->insert('studentfeedlist_master', $data);
        
        $error = $this->db->error();
        if ($error['code'] != 0) {
            log_message('error', 'Model insert error: ' . print_r($error, true));
            return false;
        }
        
        $insert_id = $this->db->insert_id();
        
        if ($insert_id && $insert_id > 0) {
            return $insert_id;
        }
        
        return false;
    }
public function getHostelsByIds($hostelIds)
{
    if (empty($hostelIds) || !is_array($hostelIds)) {
        return [];
    }
    
    // Debug: Check input
    log_message('debug', 'getHostelsByIds input: ' . print_r($hostelIds, true));
    
    $this->db->select('*');
    $this->db->from('hostel_master');
    $this->db->where_in('Hostel_id', $hostelIds);
    $this->db->where('is_active', 1);
    $this->db->order_by('Hostel_Name', 'ASC');
    
    $query = $this->db->get();
    $result = $query->result();
    
    // Debug: Check SQL query and result
    log_message('debug', 'SQL Query: ' . $this->db->last_query());
    log_message('debug', 'getHostelsByIds result: ' . print_r($result, true));
    
    return $result;
}
    /**
     * Update student feedback record
     */
    public function updateStudent($student_id, $data)
    {
        $this->db->where('Student_id', $student_id);
        return $this->db->update('studentfeedlist_master', $data);
    }

    /**
     * Get student by ID
     */
    public function getStudentById($student_id)
    {
        $this->db->select('sf.*');
        $this->db->from('studentfeedlist_master sf');
        $this->db->where('sf.Student_id', $student_id);
        $this->db->where('sf.is_active', 1);
        return $this->db->get()->row();
    }

    /**
     * Get students by group (same excel_name or related records)
     * This assumes records uploaded together have the same excel_name
     */
    public function getStudentsByGroup($student_id)
    {
        // First get the record to find its grouping criteria
        $student = $this->getStudentById($student_id);
        
        if (!$student) {
            return [];
        }

        $this->db->select('sf.*');
        $this->db->from('studentfeedlist_master sf');
        
        // Group by excel_name if it exists, otherwise by year/institute/hostel combination
        if (!empty($student->excel_name)) {
            $this->db->where('sf.excel_name', $student->excel_name);
        } else {
            $this->db->where('sf.FiniYear', $student->FiniYear);
            $this->db->where('sf.fkInsitute_id', $student->fkInsitute_id);
            $this->db->where('sf.fkHostelType_id', $student->fkHostelType_id);
            $this->db->where('(sf.excel_name IS NULL OR sf.excel_name = "")', NULL, FALSE);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->order_by('sf.Student_id', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Soft delete student record
     */
    public function softDeleteStudent($student_id)
    {
        $this->db->where('Student_id', $student_id);
        return $this->db->update('studentfeedlist_master', ['is_active' => 0]);
    }

    /**
     * Hard delete student record
     */
    public function deleteStudent($student_id)
    {
        $this->db->where('Student_id', $student_id);
        return $this->db->delete('studentfeedlist_master');
    }

    /**
     * Delete all students in a group
     */
    public function deleteStudentGroup($student_id)
    {
        // Get the group
        $students = $this->getStudentsByGroup($student_id);
        
        if (empty($students)) {
            return false;
        }

        // Collect all Student_ids
        $student_ids = [];
        foreach ($students as $student) {
            $student_ids[] = $student->Student_id;
        }

        // Soft delete all records
        $this->db->where_in('Student_id', $student_ids);
        return $this->db->update('studentfeedlist_master', ['is_active' => 0]);
    }

    /**
     * Check if PRN exists
     */
    public function checkPRNExists($prn, $excludeStudentId = null)
    {
        $this->db->where('PRN', $prn);
        $this->db->where('is_active', 1);
        
        if ($excludeStudentId) {
            $this->db->where('Student_id !=', $excludeStudentId);
        }
        
        $query = $this->db->get('studentfeedlist_master');
        return $query->num_rows() > 0;
    }

    // ==================== LIST VIEW FUNCTIONS ====================
    
    /**
     * Get all student feedback records for list view
     */
    public function getDetailView()
    {
        $this->db->select('
            sf.*,
            fm.fini_Label,
            im.Institu_name,
            hm.Hostel_Name
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'sf.FiniYear = fm.finiYear_id', 'left');
        $this->db->join('institute_master im', 'sf.fkInsitute_id = im.Institu_id', 'left');
        $this->db->join('hostel_master hm', 'sf.fkHostelType_id = hm.Hostel_id', 'left');
        $this->db->where('sf.is_active', 1);
        $this->db->order_by('sf.Student_id', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get student feedback data with filters
     */
    public function getStudentDataFiltered($filters = [])
    {
        $this->db->select('
            sf.*,
            fm.fini_Label as year_label,
            im.Institu_name,
            hm.Hostel_Name
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        $this->db->join('institute_master im', 'im.Institu_id = sf.fkInsitute_id', 'left');
        $this->db->join('hostel_master hm', 'hm.Hostel_id = sf.fkHostelType_id', 'left');
        
        if (!empty($filters['year_ids']) && is_array($filters['year_ids'])) {
            $this->db->where_in('sf.FiniYear', $filters['year_ids']);
        }
        
        if (!empty($filters['institute_ids']) && is_array($filters['institute_ids'])) {
            $this->db->where_in('sf.fkInsitute_id', $filters['institute_ids']);
        }
        
        if (!empty($filters['hostel_ids']) && is_array($filters['hostel_ids'])) {
            $this->db->where_in('sf.fkHostelType_id', $filters['hostel_ids']);
        }
        
        if (!empty($filters['branch'])) {
            $this->db->like('sf.Branch', $filters['branch']);
        }
        
        if (!empty($filters['class'])) {
            $this->db->like('sf.Class', $filters['class']);
        }
        
        if (!empty($filters['search'])) {
            $search = $filters['search'];
            $this->db->group_start();
            $this->db->like('sf.PRN', $search);
            $this->db->or_like('sf.Student_Name', $search);
            $this->db->or_like('sf.Branch', $search);
            $this->db->or_like('sf.Class', $search);
            $this->db->or_like('sf.ContactNO', $search);
            $this->db->group_end();
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->order_by('sf.Student_id', 'DESC');
        
        return $this->db->get()->result();
    }

    // ==================== STATISTICS FUNCTIONS ====================
    
    /**
     * Get student feedback statistics
     */
    public function getStudentStats($filters = [])
    {
        $this->db->select('
            COUNT(sf.Student_id) as total_students,
            COUNT(DISTINCT sf.Branch) as total_branches,
            COUNT(DISTINCT sf.Class) as total_classes,
            fm.fini_Label as year_label
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        
        if (!empty($filters['year_ids']) && is_array($filters['year_ids'])) {
            $this->db->where_in('sf.FiniYear', $filters['year_ids']);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->group_by('sf.FiniYear, fm.fini_Label');
        
        return $this->db->get()->result();
    }

    /**
     * Get branch-wise count
     */
    public function getBranchwiseCount($year_ids = [])
    {
        $this->db->select('
            sf.Branch,
            COUNT(sf.Student_id) as student_count,
            fm.fini_Label as year_label
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        
        if (!empty($year_ids) && is_array($year_ids)) {
            $this->db->where_in('sf.FiniYear', $year_ids);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->where('sf.Branch IS NOT NULL');
        $this->db->where('sf.Branch !=', '');
        $this->db->group_by('sf.Branch, fm.fini_Label');
        $this->db->order_by('student_count', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get class-wise count
     */
    public function getClasswiseCount($year_ids = [])
    {
        $this->db->select('
            sf.Class,
            COUNT(sf.Student_id) as student_count,
            fm.fini_Label as year_label
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        
        if (!empty($year_ids) && is_array($year_ids)) {
            $this->db->where_in('sf.FiniYear', $year_ids);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->where('sf.Class IS NOT NULL');
        $this->db->where('sf.Class !=', '');
        $this->db->group_by('sf.Class, fm.fini_Label');
        $this->db->order_by('student_count', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get hostel-wise count
     */
    public function getHostelwiseCount($year_ids = [])
    {
        $this->db->select('
            hm.Hostel_Name,
            COUNT(sf.Student_id) as student_count,
            fm.fini_Label as year_label
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        $this->db->join('hostel_master hm', 'hm.Hostel_id = sf.fkHostelType_id', 'left');
        
        if (!empty($year_ids) && is_array($year_ids)) {
            $this->db->where_in('sf.FiniYear', $year_ids);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->group_by('hm.Hostel_Name, fm.fini_Label');
        $this->db->order_by('student_count', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get division-wise count
     */
    public function getDivisionwiseCount($year_ids = [], $class = null)
    {
        $this->db->select('
            sf.Division,
            sf.Class,
            COUNT(sf.Student_id) as student_count,
            fm.fini_Label as year_label
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        
        if (!empty($year_ids) && is_array($year_ids)) {
            $this->db->where_in('sf.FiniYear', $year_ids);
        }
        
        if ($class) {
            $this->db->where('sf.Class', $class);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->where('sf.Division IS NOT NULL');
        $this->db->where('sf.Division !=', '');
        $this->db->group_by('sf.Division, sf.Class, fm.fini_Label');
        $this->db->order_by('sf.Class', 'ASC');
        $this->db->order_by('sf.Division', 'ASC');
        
        return $this->db->get()->result();
    }

    // ==================== YEAR FUNCTIONS ====================
    
    /**
     * Get active year
     */
    public function getActiveYear()
    {
        $this->db->select('finiYear_id, fini_Label');
        $this->db->from('finiyear_master');
        $this->db->where('Active_Deactive', 1);
        return $this->db->get()->row();
    }

    /**
     * Get all years
     */
    public function getYears()
    {
        $this->db->select('finiYear_id, fini_Label');
        $this->db->from('finiyear_master');
        $this->db->order_by('finiYear_id', 'DESC');
        return $this->db->get()->result();
    }

    /**
     * Get year by ID
     */
    public function getYearById($yearId)
    {
        $this->db->where('finiYear_id', $yearId);
        $query = $this->db->get('finiyear_master');
        return $query->row();
    }

    // ==================== INSTITUTE FUNCTIONS ====================
    
    /**
     * Get all institutes
     */
    public function getInstitute()
    {
        $this->db->select('institute_master.*');
        $this->db->from('institute_master');
        // $this->db->order_by('Institu_name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Get institute by ID
     */
    public function getInstituteById($instituteId)
    {
        $this->db->where('Institu_id', $instituteId);
        return $this->db->get('institute_master')->row();
    }

    // ==================== HOSTEL FUNCTIONS ====================
    
    /**
     * Get all hostels
     */
    public function getHostel()
    {
        $this->db->select('hostel_master.*');
        $this->db->from('hostel_master');
        $this->db->order_by('Hostel_Name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Get hostel by ID
     */
    public function getHostelById($hostelId)
    {
        $this->db->where('Hostel_id', $hostelId);
        return $this->db->get('hostel_master')->row();
    }

    // ==================== SEARCH & AUTOCOMPLETE FUNCTIONS ====================
    
    /**
     * Search students by PRN
     */
    public function searchByPRN($prn)
    {
        $this->db->select('sf.*, fm.fini_Label, im.Institu_name, hm.Hostel_Name');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        $this->db->join('institute_master im', 'im.Institu_id = sf.fkInsitute_id', 'left');
        $this->db->join('hostel_master hm', 'hm.Hostel_id = sf.fkHostelType_id', 'left');
        $this->db->like('sf.PRN', $prn);
        $this->db->where('sf.is_active', 1);
        $this->db->limit(10);
        
        return $this->db->get()->result();
    }

    /**
     * Search students by name
     */
    public function searchByName($name)
    {
        $this->db->select('sf.*, fm.fini_Label, im.Institu_name, hm.Hostel_Name');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        $this->db->join('institute_master im', 'im.Institu_id = sf.fkInsitute_id', 'left');
        $this->db->join('hostel_master hm', 'hm.Hostel_id = sf.fkHostelType_id', 'left');
        $this->db->like('sf.Student_Name', $name);
        $this->db->where('sf.is_active', 1);
        $this->db->limit(10);
        
        return $this->db->get()->result();
    }

    /**
     * Get unique branches
     */
    public function getUniqueBranches()
    {
        $this->db->select('DISTINCT Branch');
        $this->db->from('studentfeedlist_master');
        $this->db->where('Branch IS NOT NULL');
        $this->db->where('Branch !=', '');
        $this->db->where('is_active', 1);
        $this->db->order_by('Branch', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get unique classes
     */
    public function getUniqueClasses()
    {
        $this->db->select('DISTINCT Class');
        $this->db->from('studentfeedlist_master');
        $this->db->where('Class IS NOT NULL');
        $this->db->where('Class !=', '');
        $this->db->where('is_active', 1);
        $this->db->order_by('Class', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get unique divisions
     */
    public function getUniqueDivisions($class = null)
    {
        $this->db->select('DISTINCT Division, Class');
        $this->db->from('studentfeedlist_master');
        
        if ($class) {
            $this->db->where('Class', $class);
        }
        
        $this->db->where('Division IS NOT NULL');
        $this->db->where('Division !=', '');
        $this->db->where('is_active', 1);
        $this->db->order_by('Class', 'ASC');
        $this->db->order_by('Division', 'ASC');
        
        return $this->db->get()->result();
    }

    // ==================== EXPORT FUNCTIONS ====================
    
    /**
     * Get data for Excel export
     */
    public function getExportData($filters = [])
    {
        $this->db->select('
            sf.PRN,
            sf.Student_Name,
            sf.Class,
            sf.Branch,
            sf.Division,
            sf.ContactNO,
            fm.fini_Label as Academic_Year,
            im.Institu_name as Institute,
            hm.Hostel_Name as Hostel_Type
        ');
        $this->db->from('studentfeedlist_master sf');
        $this->db->join('finiyear_master fm', 'fm.finiYear_id = sf.FiniYear', 'left');
        $this->db->join('institute_master im', 'im.Institu_id = sf.fkInsitute_id', 'left');
        $this->db->join('hostel_master hm', 'hm.Hostel_id = sf.fkHostelType_id', 'left');
        
        if (!empty($filters['year_ids']) && is_array($filters['year_ids'])) {
            $this->db->where_in('sf.FiniYear', $filters['year_ids']);
        }
        
        if (!empty($filters['institute_ids']) && is_array($filters['institute_ids'])) {
            $this->db->where_in('sf.fkInsitute_id', $filters['institute_ids']);
        }
        
        if (!empty($filters['hostel_ids']) && is_array($filters['hostel_ids'])) {
            $this->db->where_in('sf.fkHostelType_id', $filters['hostel_ids']);
        }
        
        $this->db->where('sf.is_active', 1);
        $this->db->order_by('sf.Student_id', 'DESC');
        
        return $this->db->get()->result();
    }
}