<?php
class User_model extends CI_Model {
    public function __construct()
    {
        $this->load->database();
    }

    // Check if username exists for the same faculty
    public function checkUsernameExists($username, $faculty_id, $user_id = null)
    {
        $this->db->where('UserName', $username);
        $this->db->where('fkfaculty_ID', $faculty_id);
        
        // Exclude current user when updating
        if ($user_id) {
            $this->db->where('UserID !=', $user_id);
        }
        
        $query = $this->db->get('user_master');
        return $query->num_rows() > 0;
    }

    // Insert user data
    public function insertdata($model)
    {
        return $this->db->insert('user_master', $model);
    }

    // Insert user access (menu and submenu)
    public function insertUserAccess($user_id, $menu_ids = [], $submenu_ids = [])
    {
        $this->db->trans_start();
        
        // First, get menu_id for each submenu to maintain proper relationships
        $submenu_menu_map = [];
        if (!empty($submenu_ids)) {
            $this->db->select('SubID, fkMenuID');
            $this->db->from('submenu_master');
            $this->db->where_in('SubID', $submenu_ids);
            $result = $this->db->get()->result();
            
            foreach ($result as $row) {
                $submenu_menu_map[$row->SubID] = $row->fkMenuID;
            }
        }
        
        // Combine menus from direct selection and from submenus
        $all_menu_ids = array_unique(array_merge(
            (array)$menu_ids, 
            array_values($submenu_menu_map)
        ));
        
        // Insert menu access
        if (!empty($all_menu_ids)) {
            foreach ($all_menu_ids as $menu_id) {
                // Check if already exists
                $exists = $this->db->where('fkUserID', $user_id)
                                   ->where('fkMenu_Id', $menu_id)
                                   ->where('fkSubMenu_Id IS NULL')
                                   ->get('useraccess_master')
                                   ->num_rows();
                
                if ($exists == 0) {
                    $data = [
                        'fkUserID' => $user_id,
                        'fkMenu_Id' => $menu_id,
                        'fkSubMenu_Id' => NULL
                    ];
                    $this->db->insert('useraccess_master', $data);
                }
            }
        }
        
        // Insert submenu access with proper menu relationship
        if (!empty($submenu_ids)) {
            foreach ($submenu_ids as $submenu_id) {
                $menu_id = isset($submenu_menu_map[$submenu_id]) ? $submenu_menu_map[$submenu_id] : NULL;
                
                // Check if already exists
                $exists = $this->db->where('fkUserID', $user_id)
                                   ->where('fkSubMenu_Id', $submenu_id)
                                   ->get('useraccess_master')
                                   ->num_rows();
                
                if ($exists == 0) {
                    $data = [
                        'fkUserID' => $user_id,
                        'fkMenu_Id' => $menu_id, // Keep menu reference for submenus
                        'fkSubMenu_Id' => $submenu_id
                    ];
                    $this->db->insert('useraccess_master', $data);
                }
            }
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // Update user access
    public function updateUserAccess($user_id, $menu_ids = [], $submenu_ids = [])
    {
        $this->db->trans_start();
        
        // Delete existing access
        $this->db->where('fkUserID', $user_id);
        $this->db->delete('useraccess_master');
        
        // Insert new access
        $this->insertUserAccess($user_id, $menu_ids, $submenu_ids);
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // Get faculty list
    public function getfaculty()
    {
        $this->db->select('fac_id, fac_Fullname');
        $this->db->from('faculty_master');
        $query = $this->db->get();
        return $query->result();
    }

    // Get role list
    public function getRole()
    {
        $this->db->select('Role_id, Role_Name');
        $this->db->from('role_master');
        $query = $this->db->get();
        return $query->result();
    }

    // Get branch list
    public function getBranch()
    {
        $this->db->select('branch_id, branch_name');
        $this->db->from('branch_master');
        $query = $this->db->get();
        return $query->result();
    }

    // Get all menus with submenus (hierarchical structure)
    public function getMenusWithSubmenus()
    {
        // Get all menus
        $this->db->select('MenuID, MenuName, MenuURL, MenuIcon, ArrangeNo');
        $this->db->from('menu_master');
        $this->db->where('is_active', 1);
        $menus = $this->db->get()->result();

        // Get submenus for each menu
        foreach ($menus as &$menu) {
            $this->db->select('SubID, Name, SubName, URL, ArrangeNO');
            $this->db->from('submenu_master');
            $this->db->where('fkMenuID', $menu->MenuID);
            $this->db->where('is_active', 1);
            $menu->submenus = $this->db->get()->result();
        }

        return $menus;
    }

    // Get user's assigned menu and submenu access
    public function getUserMenuAccess($user_id)
    {
        // Get menu access - menus that are explicitly selected OR have selected submenus
        $this->db->distinct();
        $this->db->select('fkMenu_Id');
        $this->db->from('useraccess_master');
        $this->db->where('fkUserID', $user_id);
        $this->db->where('fkMenu_Id IS NOT NULL');
        $menu_result = $this->db->get()->result();
        $menus = array_column($menu_result, 'fkMenu_Id');

        // Get submenu access
        $this->db->distinct();
        $this->db->select('fkSubMenu_Id');
        $this->db->from('useraccess_master');
        $this->db->where('fkUserID', $user_id);
        $this->db->where('fkSubMenu_Id IS NOT NULL');
        $submenu_result = $this->db->get()->result();
        $submenus = array_column($submenu_result, 'fkSubMenu_Id');

        return [
            'menus' => array_values(array_unique($menus)),
            'submenus' => array_values(array_unique($submenus))
        ];
    }

    // Get user by ID
    public function getbyid($id)
    {
        $this->db->select("user_master.*");
        $this->db->where('UserID', $id);
        $query = $this->db->get('user_master');
        return $query->result();
    }

    // Update user data
    public function update($model)
    {
        return $this->db->where('UserID', $model['UserID'])->update('user_master', $model);
    }

    // Get all users with details
    public function getdetailview()
    {
        $this->db->select('
            user_master.*,
            faculty_master.fac_Fullname,
            role_master.Role_Name
        ');
        $this->db->from('user_master');
        $this->db->join('faculty_master', 'faculty_master.fac_id = user_master.fkfaculty_ID', 'left');
        $this->db->join('role_master', 'role_master.Role_id = user_master.fkRole_id', 'left');
        $query = $this->db->get();
        
        // Process results to get branch names
        $results = $query->result();
        foreach ($results as &$user) {
            if (!empty($user->fkBranch_Id)) {
                $branch_ids = explode(',', $user->fkBranch_Id);
                $this->db->select('branch_name');
                $this->db->from('branch_master');
                $this->db->where_in('branch_id', $branch_ids);
                $branches = $this->db->get()->result();
                $user->branch_names = implode(', ', array_column($branches, 'branch_name'));
            } else {
                $user->branch_names = '';
            }
        }
        
        return $results;
    }

    // Check if user has menu access
    public function hasMenuAccess($user_id, $menu_id)
    {
        $this->db->where('fkUserID', $user_id);
        $this->db->where('fkMenu_Id', $menu_id);
        return $this->db->get('useraccess_master')->num_rows() > 0;
    }

    // Check if user has submenu access
    public function hasSubmenuAccess($user_id, $submenu_id)
    {
        $this->db->where('fkUserID', $user_id);
        $this->db->where('fkSubMenu_Id', $submenu_id);
        return $this->db->get('useraccess_master')->num_rows() > 0;
    }

    // Get user's navigation menus (for sidebar/menu display)
    public function getUserNavigationMenus($user_id)
    {
        $this->db->distinct();
        $this->db->select('m.MenuID, m.MenuName, m.MenuURL, m.MenuIcon, m.ArrangeNo');
        $this->db->from('menu_master m');
        $this->db->join('useraccess_master ua', 'ua.fkMenu_Id = m.MenuID');
        $this->db->where('ua.fkUserID', $user_id);
        $this->db->where('m.is_default', 1);
        $this->db->where('m.is_on', 1);
        
        $menus = $this->db->get()->result();

        foreach ($menus as &$menu) {
            $this->db->distinct();
            $this->db->select('sm.SubID, sm.Name, sm.SubName, sm.URL, sm.ArrangeNO');
            $this->db->from('submenu_master sm');
            $this->db->join('useraccess_master ua', 'ua.fkSubMenu_Id = sm.SubID');
            $this->db->where('ua.fkUserID', $user_id);
            $this->db->where('sm.fkMenuID', $menu->MenuID);
            $this->db->where('sm.is_default', 1);
            $menu->submenus = $this->db->get()->result();
        }

        return $menus;
    }
}